<?php
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://codecanyon.net/user/kamleshyadav
 * @since      1.0.0
 *
 * @package    Astro
 * @subpackage Astro/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Astro
 * @subpackage Astro/includes
 * @author     kamleshyadav <kamlesh.yadav@himanshusofttech.com >
 */
class Astro {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Astro_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'Astro_VERSION' ) ) {
			$this->version = Astro_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'astro';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Astro_Loader. Orchestrates the hooks of the plugin.
	 * - Astro_i18n. Defines internationalization functionality.
	 * - Astro_Admin. Defines all hooks for the admin area.
	 * - Astro_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-astro-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-astro-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-astro-admin.php';
        
		/**
		 * Astro Hotel Metabox file load
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/appointment/astro-appointment-metabox.php';   
        
		/**
		 * Astro Appointment Metabox 
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/hotel/astro-hotel-metabox.php';

		/**
		 * Astro Event Metabox 
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/event/astro-event-metabox.php'; 

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-astro-public.php';
		
		/**
		 * Astro Email Template Load
		 */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'email/astro-email-template.php';

		$this->loader = new Astro_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Astro_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Astro_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

	  $plugin_admin = new Astro_Admin( $this->get_plugin_name(), $this->get_version() );
      $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
	  $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
      $this->loader->add_action( 'admin_menu', $plugin_admin, 'Astro_admin_menu');
	  $this->loader->add_action('init', $plugin_admin, 'Astro_custom_post_type');
      $this->loader->add_action( 'wp_ajax_nopriv_Astro_connection', $plugin_admin, 'Astro_connection');
	  $this->loader->add_action( 'wp_ajax_Astro_connection', $plugin_admin, 'Astro_connection');

	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_disconnection', $plugin_admin, 'Astro_disconnection');
	  $this->loader->add_action( 'wp_ajax_Astro_disconnection', $plugin_admin, 'Astro_disconnection');

	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_hotel_stripe_payment_details_save', $plugin_admin, 'Astro_hotel_stripe_payment_details_save');
	  $this->loader->add_action( 'wp_ajax_Astro_hotel_stripe_payment_details_save', $plugin_admin, 'Astro_hotel_stripe_payment_details_save');
      
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_cafe_payment_dateails_save', $plugin_admin, 'Astro_cafe_payment_dateails_save');
	  $this->loader->add_action( 'wp_ajax_Astro_cafe_payment_dateails_save', $plugin_admin, 'Astro_cafe_payment_dateails_save');
	  
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_cafe_custom_option_save', $plugin_admin, 'Astro_cafe_custom_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_cafe_custom_option_save', $plugin_admin, 'Astro_cafe_custom_option_save');
       
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_hotel_custom_option_save', $plugin_admin, 'Astro_hotel_custom_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_hotel_custom_option_save', $plugin_admin, 'Astro_hotel_custom_option_save');

      $this->loader->add_action( 'wp_ajax_nopriv_Astro_appointment_paymentdetail_save', $plugin_admin, 'Astro_appointment_paymentdetail_save');
	  $this->loader->add_action( 'wp_ajax_Astro_appointment_paymentdetail_save', $plugin_admin, 'Astro_appointment_paymentdetail_save');
	   
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_appointment_custom_option_save', $plugin_admin, 'Astro_appointment_custom_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_appointment_custom_option_save', $plugin_admin, 'Astro_appointment_custom_option_save');

	  $this->loader->add_action('wp_ajax_nopriv_Astro_qr_code_save', $plugin_admin, 'Astro_qr_code_save');
	  $this->loader->add_action('wp_ajax_Astro_qr_code_save', $plugin_admin, 'Astro_qr_code_save');
	  
	  
	  $this->loader->add_action('wp_ajax_nopriv_Astro_appointment_free_payment', $plugin_admin, 'Astro_appointment_free_payment');
	  $this->loader->add_action('wp_ajax_Astro_appointment_free_payment', $plugin_admin, 'Astro_appointment_free_payment');
	  
	  

      /**
	   * Astro Hotel Metaoption Class
	   */
	  $hotel_meta_option = new Hotel_metabox( $this->get_plugin_name(), $this->get_version() );
	  $this->loader->add_action('admin_menu', $hotel_meta_option, 'hotel_add_meta_boxes'); 
	  $this->loader->add_action('save_post', $hotel_meta_option, 'hetal_save_meta_options'); 
      
	  /**
	   * Astro Appointment Metaoption
	   */
	  $appointment_meta_option = new Appointment_metabox($this->get_plugin_name(),$this->get_version());
      $this->loader->add_action('admin_menu',$appointment_meta_option,'appointment_add_meta_boxes');
	  $this->loader->add_action('save_post',$appointment_meta_option,'appointment_save_meta_options');
   
	  /**
	   * Astro Event Metaoption 
	   */
	  $appointment_meta_option = new Cafe_metabox($this->get_plugin_name(),$this->get_version());
      $this->loader->add_action('admin_menu',$appointment_meta_option,'cafe_add_meta_boxes');
	  $this->loader->add_action('save_post',$appointment_meta_option,'cafe_save_meta_options');
	  
	  /**
	   * Astro Payment Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_payment_details_save', $plugin_admin, 'Astro_payment_details_save');
	  $this->loader->add_action( 'wp_ajax_Astro_payment_details_save', $plugin_admin, 'Astro_payment_details_save');
	  
	  /**
	   * Astro Payment Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_hotel_razorpay_payment_details_save', $plugin_admin, 'Astro_hotel_razorpay_payment_details_save');
	  $this->loader->add_action( 'wp_ajax_Astro_hotel_razorpay_payment_details_save', $plugin_admin, 'Astro_hotel_razorpay_payment_details_save');
	  
	  /**
	   * Astro sms Hotel Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_hotel_sms_option_save', $plugin_admin, 'Astro_hotel_sms_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_hotel_sms_option_save', $plugin_admin, 'Astro_hotel_sms_option_save');
	  
	  /**
	   * Astro sms Hotel Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_hotel_email_option_save', $plugin_admin, 'Astro_hotel_email_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_hotel_email_option_save', $plugin_admin, 'Astro_hotel_email_option_save');
	  
	  /**
	   * Astro Hotel color Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_hotel_color_option_save', $plugin_admin, 'Astro_hotel_color_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_hotel_color_option_save', $plugin_admin, 'Astro_hotel_color_option_save');
	  
	  /**
	   * Astro event Payment Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_payment_cafe_details_save', $plugin_admin, 'Astro_payment_cafe_details_save');
	  $this->loader->add_action( 'wp_ajax_Astro_payment_cafe_details_save', $plugin_admin, 'Astro_payment_cafe_details_save');
	  
	  /**
	   * Astro event Razorpay Payment Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_cafe_razorpay_payment_details_save', $plugin_admin, 'Astro_cafe_razorpay_payment_details_save');
	  $this->loader->add_action( 'wp_ajax_Astro_cafe_razorpay_payment_details_save', $plugin_admin, 'Astro_cafe_razorpay_payment_details_save');
	  
	  
	  /**
	   * Astro sms event Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_cafe_sms_option_save', $plugin_admin, 'Astro_cafe_sms_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_cafe_sms_option_save', $plugin_admin, 'Astro_cafe_sms_option_save');
	  
	  /**
	   * Astro Email event Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_cafe_email_option_save', $plugin_admin, 'Astro_cafe_email_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_cafe_email_option_save', $plugin_admin, 'Astro_cafe_email_option_save');
	  
	  /**
	   * Astro Color event Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_cafe_color_option_save', $plugin_admin, 'Astro_cafe_color_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_cafe_color_option_save', $plugin_admin, 'Astro_cafe_color_option_save');
	  
	  /**
	   * Astro Color event Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_app_payment_save', $plugin_admin, 'Astro_app_payment_save');
	  $this->loader->add_action( 'wp_ajax_Astro_app_payment_save', $plugin_admin, 'Astro_app_payment_save');
	  
	  /**
	   * Astro Color event Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_app_razorpay_save', $plugin_admin, 'Astro_app_razorpay_save');
	  $this->loader->add_action( 'wp_ajax_Astro_app_razorpay_save', $plugin_admin, 'Astro_app_razorpay_save');
	  
	  /**
	   * Astro Color event Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_app_sms_save', $plugin_admin, 'Astro_app_sms_save');
	  $this->loader->add_action( 'wp_ajax_Astro_app_sms_save', $plugin_admin, 'Astro_app_sms_save');
	  
	  /**
	   * Astro Color event Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_app_color_option_save', $plugin_admin, 'Astro_app_color_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_app_color_option_save', $plugin_admin, 'Astro_app_color_option_save');
	  
	  /**
	   * Astro Color event Setting 
	   */
	  $this->loader->add_action( 'wp_ajax_nopriv_Astro_app_email_option_save', $plugin_admin, 'Astro_app_email_option_save');
	  $this->loader->add_action( 'wp_ajax_Astro_app_email_option_save', $plugin_admin, 'Astro_app_email_option_save');
	  
	}
	 
	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Astro_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
		//$this->loader->add_filter( 'the_content', $plugin_public, 'Astro_hotel_rooms_single');

		$this->loader->add_action('wp_ajax_nopriv_Astro_hotel_room_filter', $plugin_public, 'Astro_hotel_room_filter');
	    $this->loader->add_action('wp_ajax_Astro_hotel_room_filter', $plugin_public, 'Astro_hotel_room_filter');

		$this->loader->add_action( 'wp_ajax_nopriv_Astro_hotel_singel_page_checkdate', $plugin_public, 'Astro_hotel_singel_page_checkdate');
	    $this->loader->add_action( 'wp_ajax_Astro_hotel_singel_page_checkdate', $plugin_public, 'Astro_hotel_singel_page_checkdate');

        $this->loader->add_action( 'wp_ajax_nopriv_Astro_hotel_stripe_payment', $plugin_public, 'Astro_hotel_stripe_payment');
	    $this->loader->add_action( 'wp_ajax_Astro_hotel_stripe_payment', $plugin_public, 'Astro_hotel_stripe_payment');

		$this->loader->add_action('wp_ajax_nopriv_Astro_appointment_schedule_show', $plugin_public, 'Astro_appointment_schedule_show');
	    $this->loader->add_action('wp_ajax_Astro_appointment_schedule_show', $plugin_public, 'Astro_appointment_schedule_show');

		$this->loader->add_action('wp_ajax_nopriv_Astro_appointment_stripe_payment', $plugin_public, 'Astro_appointment_stripe_payment');
	    $this->loader->add_action('wp_ajax_Astro_appointment_stripe_payment', $plugin_public, 'Astro_appointment_stripe_payment');
		
    }

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Astro_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}