<?php
session_start();
require_once '../app/Config/Config.php';
require_once '../app/Config/Database.php';
require_once '../app/Models/User.php';
require_once '../app/Models/Form.php';
require_once '../app/Models/Submission.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();
$user = new User($db);
$form = new Form($db);
$submission = new Submission($db);

$user->getById($_SESSION['user_id']);

// Get filter parameters
$form_id = $_GET['form_id'] ?? '';
$search = $_GET['search'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$page = max(1, intval($_GET['page'] ?? 1));
$per_page = 20;

// Get user's forms for filter dropdown
$user_forms = $form->getByUserId($_SESSION['user_id']);

// Build query for submissions
$where_conditions = ['f.user_id = :user_id'];
$params = [':user_id' => $_SESSION['user_id']];

if ($form_id) {
    $where_conditions[] = 's.form_id = :form_id';
    $params[':form_id'] = $form_id;
}

if ($search) {
    $where_conditions[] = 's.submission_data LIKE :search';
    $params[':search'] = '%' . $search . '%';
}

if ($date_from) {
    $where_conditions[] = 'DATE(s.created_at) >= :date_from';
    $params[':date_from'] = $date_from;
}

if ($date_to) {
    $where_conditions[] = 'DATE(s.created_at) <= :date_to';
    $params[':date_to'] = $date_to;
}

$where_clause = 'WHERE ' . implode(' AND ', $where_conditions);

// Get total count
$count_query = "SELECT COUNT(*) as total FROM submissions s 
                JOIN forms f ON s.form_id = f.id 
                {$where_clause}";
$count_stmt = $db->prepare($count_query);
$count_stmt->execute($params);
$total_submissions = $count_stmt->fetch(PDO::FETCH_ASSOC)['total'];

// Calculate pagination
$total_pages = ceil($total_submissions / $per_page);
$offset = ($page - 1) * $per_page;

// Get submissions
$submissions_query = "SELECT s.*, f.form_name, f.form_endpoint 
                     FROM submissions s 
                     JOIN forms f ON s.form_id = f.id 
                     {$where_clause}
                     ORDER BY s.created_at DESC 
                     LIMIT {$per_page} OFFSET {$offset}";
$submissions_stmt = $db->prepare($submissions_query);
$submissions_stmt->execute($params);
$submissions = $submissions_stmt->fetchAll(PDO::FETCH_ASSOC);

// Handle export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    exportToCSV($submissions, $db, $where_clause, $params);
    exit();
}

function exportToCSV($submissions, $db, $where_clause, $params) {
    // Get all submissions for export (without pagination)
    $export_query = "SELECT s.*, f.form_name 
                    FROM submissions s 
                    JOIN forms f ON s.form_id = f.id 
                    {$where_clause}
                    ORDER BY s.created_at DESC";
    $export_stmt = $db->prepare($export_query);
    $export_stmt->execute($params);
    $all_submissions = $export_stmt->fetchAll(PDO::FETCH_ASSOC);

    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="submissions_' . date('Y-m-d') . '.csv"');

    $output = fopen('php://output', 'w');
    
    // Write header
    if (!empty($all_submissions)) {
        $first_submission = json_decode($all_submissions[0]['submission_data'], true);
        $headers = array_merge(['Form Name', 'Submitted At', 'IP Address'], array_keys($first_submission));
        fputcsv($output, $headers);

        // Write data
        foreach ($all_submissions as $submission) {
            $data = json_decode($submission['submission_data'], true);
            $row = array_merge([
                $submission['form_name'],
                $submission['created_at'],
                $submission['ip_address']
            ], array_values($data));
            fputcsv($output, $row);
        }
    }

    fclose($output);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Submissions - SmartForm</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand fw-bold" href="index.php"><i class="fas fa-bolt"></i> SmartForm</a>
            <div class="navbar-nav ms-auto">
                <div class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                        <i class="fas fa-user-circle me-1"></i> <?php echo htmlspecialchars($user->username); ?>
                    </a>
                    <ul class="dropdown-menu">
                        <li><a class="dropdown-item" href="profile.php"><i class="fas fa-user me-2"></i>Profile</a></li>
                        <li><a class="dropdown-item" href="billing.php"><i class="fas fa-credit-card me-2"></i>Billing</a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
                    </ul>
                </div>
            </div>
        </div>
    </nav>

    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 dashboard-sidebar p-0">
                <div class="p-3">
                    <ul class="nav flex-column dashboard-nav">
                        <li class="nav-item">
                            <a class="nav-link" href="dashboard.php">
                                <i class="fas fa-tachometer-alt me-2"></i>Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="forms.php">
                                <i class="fas fa-wpforms me-2"></i>Forms
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link active" href="submissions.php">
                                <i class="fas fa-paper-plane me-2"></i>Submissions
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="analytics.php">
                                <i class="fas fa-chart-bar me-2"></i>Analytics
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="integrations.php">
                                <i class="fas fa-plug me-2"></i>Integrations
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="settings.php">
                                <i class="fas fa-cog me-2"></i>Settings
                            </a>
                        </li>
                    </ul>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 col-lg-10">
                <div class="p-4">
                    <!-- Header -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                            <h1 class="h3 mb-0">Submissions</h1>
                            <p class="text-muted">View and manage form submissions</p>
                        </div>
                        <div class="btn-group">
                            <a href="<?php echo $_SERVER['REQUEST_URI'] . (strpos($_SERVER['REQUEST_URI'], '?') ? '&' : '?') . 'export=csv'; ?>" class="btn btn-outline-primary">
                                <i class="fas fa-download me-2"></i>Export CSV
                            </a>
                        </div>
                    </div>

                    <!-- Filters -->
                    <div class="card mb-4">
                        <div class="card-body">
                            <form method="GET" action="" class="row g-3">
                                <div class="col-md-3">
                                    <label for="form_id" class="form-label">Form</label>
                                    <select class="form-select" id="form_id" name="form_id">
                                        <option value="">All Forms</option>
                                        <?php foreach ($user_forms as $user_form): ?>
                                            <option value="<?php echo $user_form['id']; ?>" <?php echo $form_id == $user_form['id'] ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($user_form['form_name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label for="search" class="form-label">Search</label>
                                    <input type="text" class="form-control" id="search" name="search" value="<?php echo htmlspecialchars($search); ?>" placeholder="Search submissions...">
                                </div>
                                <div class="col-md-2">
                                    <label for="date_from" class="form-label">From Date</label>
                                    <input type="date" class="form-control" id="date_from" name="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                                </div>
                                <div class="col-md-2">
                                    <label for="date_to" class="form-label">To Date</label>
                                    <input type="date" class="form-control" id="date_to" name="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">&nbsp;</label>
                                    <div class="d-grid">
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-search me-2"></i>Filter
                                        </button>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Results -->
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5 class="mb-0">
                                Submissions 
                                <span class="badge bg-secondary"><?php echo number_format($total_submissions); ?></span>
                            </h5>
                            <?php if ($total_submissions > 0): ?>
                                <small class="text-muted">
                                    Showing <?php echo number_format($offset + 1); ?>-<?php echo number_format(min($offset + $per_page, $total_submissions)); ?> 
                                    of <?php echo number_format($total_submissions); ?>
                                </small>
                            <?php endif; ?>
                        </div>
                        <div class="card-body p-0">
                            <?php if (empty($submissions)): ?>
                                <div class="text-center py-5">
                                    <i class="fas fa-paper-plane fa-3x text-muted mb-3"></i>
                                    <h5>No submissions found</h5>
                                    <p class="text-muted">
                                        <?php if ($form_id || $search || $date_from || $date_to): ?>
                                            Try adjusting your filters to see more results.
                                        <?php else: ?>
                                            Submissions will appear here once your forms start receiving data.
                                        <?php endif; ?>
                                    </p>
                                </div>
                            <?php else: ?>
                                <div class="table-responsive">
                                    <table class="table table-hover mb-0">
                                        <thead class="table-light">
                                            <tr>
                                                <th>Form</th>
                                                <th>Data</th>
                                                <th>Submitted</th>
                                                <th>IP Address</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($submissions as $sub): ?>
                                                <?php $submission_data = json_decode($sub['submission_data'], true); ?>
                                                <tr>
                                                    <td>
                                                        <strong><?php echo htmlspecialchars($sub['form_name']); ?></strong>
                                                        <br>
                                                        <small class="text-muted">/f/<?php echo $sub['form_endpoint']; ?></small>
                                                    </td>
                                                    <td>
                                                        <div class="submission-preview">
                                                            <?php 
                                                            $preview_count = 0;
                                                            foreach ($submission_data as $key => $value): 
                                                                if ($preview_count >= 3) break;
                                                                $preview_count++;
                                                            ?>
                                                                <div class="small">
                                                                    <strong><?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $key))); ?>:</strong>
                                                                    <?php echo htmlspecialchars(substr($value, 0, 50)) . (strlen($value) > 50 ? '...' : ''); ?>
                                                                </div>
                                                            <?php endforeach; ?>
                                                            <?php if (count($submission_data) > 3): ?>
                                                                <small class="text-muted">+<?php echo count($submission_data) - 3; ?> more fields</small>
                                                            <?php endif; ?>
                                                        </div>
                                                    </td>
                                                    <td>
                                                        <?php echo date('M j, Y g:i A', strtotime($sub['created_at'])); ?>
                                                    </td>
                                                    <td>
                                                        <code><?php echo htmlspecialchars($sub['ip_address']); ?></code>
                                                    </td>
                                                    <td>
                                                        <div class="btn-group btn-group-sm">
                                                            <button class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#submissionModal" 
                                                                    data-submission='<?php echo htmlspecialchars(json_encode($sub)); ?>'>
                                                                <i class="fas fa-eye"></i>
                                                            </button>
                                                            <button class="btn btn-outline-danger" onclick="deleteSubmission(<?php echo $sub['id']; ?>)">
                                                                <i class="fas fa-trash"></i>
                                                            </button>
                                                        </div>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>

                                <!-- Pagination -->
                                <?php if ($total_pages > 1): ?>
                                    <div class="card-footer">
                                        <nav aria-label="Submissions pagination">
                                            <ul class="pagination justify-content-center mb-0">
                                                <?php if ($page > 1): ?>
                                                    <li class="page-item">
                                                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">
                                                            <i class="fas fa-chevron-left"></i>
                                                        </a>
                                                    </li>
                                                <?php endif; ?>

                                                <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                                                    <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                                                            <?php echo $i; ?>
                                                        </a>
                                                    </li>
                                                <?php endfor; ?>

                                                <?php if ($page < $total_pages): ?>
                                                    <li class="page-item">
                                                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">
                                                            <i class="fas fa-chevron-right"></i>
                                                        </a>
                                                    </li>
                                                <?php endif; ?>
                                            </ul>
                                        </nav>
                                    </div>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Submission Detail Modal -->
    <div class="modal fade" id="submissionModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Submission Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="submissionDetails"></div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="assets/js/main.js"></script>
    <script>
        // Handle submission modal
        document.getElementById('submissionModal').addEventListener('show.bs.modal', function (event) {
            const button = event.relatedTarget;
            const submission = JSON.parse(button.getAttribute('data-submission'));
            const submissionData = JSON.parse(submission.submission_data);
            
            let html = `
                <div class="row mb-3">
                    <div class="col-md-6">
                        <strong>Form:</strong> ${submission.form_name}
                    </div>
                    <div class="col-md-6">
                        <strong>Submitted:</strong> ${new Date(submission.created_at).toLocaleString()}
                    </div>
                </div>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <strong>IP Address:</strong> <code>${submission.ip_address}</code>
                    </div>
                    <div class="col-md-6">
                        <strong>User Agent:</strong> <small>${submission.user_agent}</small>
                    </div>
                </div>
                <hr>
                <h6>Form Data:</h6>
                <div class="table-responsive">
                    <table class="table table-sm">
                        <tbody>
            `;
            
            for (const [key, value] of Object.entries(submissionData)) {
                const fieldName = key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
                html += `
                    <tr>
                        <td class="fw-bold" style="width: 30%;">${fieldName}:</td>
                        <td>${value}</td>
                    </tr>
                `;
            }
            
            html += `
                        </tbody>
                    </table>
                </div>
            `;
            
            document.getElementById('submissionDetails').innerHTML = html;
        });

        // Delete submission function
        function deleteSubmission(submissionId) {
            if (confirm('Are you sure you want to delete this submission? This action cannot be undone.')) {
                fetch('api/submissions.php', {
                    method: 'DELETE',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({ id: submissionId })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Error deleting submission: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error deleting submission');
                });
            }
        }
    </script>
</body>
</html>