<?php
session_start();
require_once '../../app/Config/Config.php';
require_once '../../app/Config/Database.php';
require_once '../../app/Models/Submission.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

$database = new Database();
$db = $database->getConnection();
$submission = new Submission($db);

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        handleGet();
        break;
    case 'DELETE':
        handleDelete();
        break;
    default:
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
        break;
}

function handleGet() {
    global $submission;
    
    $form_id = $_GET['form_id'] ?? '';
    $limit = min(100, max(1, intval($_GET['limit'] ?? 20)));
    $offset = max(0, intval($_GET['offset'] ?? 0));
    
    try {
        if ($form_id) {
            $submissions = $submission->getByFormId($form_id, $limit, $offset);
        } else {
            $submissions = $submission->getByUserId($_SESSION['user_id'], $limit, $offset);
        }
        
        echo json_encode([
            'success' => true,
            'data' => $submissions
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Failed to fetch submissions'
        ]);
    }
}

function handleDelete() {
    global $submission, $db;
    
    $input = json_decode(file_get_contents('php://input'), true);
    $submission_id = $input['id'] ?? '';
    
    if (empty($submission_id)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Submission ID required']);
        return;
    }
    
    try {
        // Verify ownership
        $query = "SELECT s.id FROM submissions s 
                  JOIN forms f ON s.form_id = f.id 
                  WHERE s.id = :id AND f.user_id = :user_id";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $submission_id);
        $stmt->bindParam(':user_id', $_SESSION['user_id']);
        $stmt->execute();
        
        if ($stmt->rowCount() === 0) {
            http_response_code(404);
            echo json_encode(['success' => false, 'message' => 'Submission not found']);
            return;
        }
        
        // Delete submission
        $submission->id = $submission_id;
        if ($submission->delete()) {
            echo json_encode(['success' => true, 'message' => 'Submission deleted successfully']);
        } else {
            throw new Exception('Failed to delete submission');
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Failed to delete submission'
        ]);
    }
}
?>