<?php

namespace SmartForm\Config;

use PDO;
use PDOException;

class Database
{
    private static $instance = null;
    private $connection;
    private $host;
    private $database;
    private $username;
    private $password;
    private $charset;
    private $options;
    
    private function __construct()
    {
        $this->host = Config::env('DB_HOST', 'localhost');
        $this->database = Config::env('DB_NAME', 'smartform');
        $this->username = Config::env('DB_USER', 'root');
        $this->password = Config::env('DB_PASS', '');
        $this->charset = 'utf8mb4';
        
        $this->options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES {$this->charset}",
            PDO::ATTR_PERSISTENT => false,
            PDO::ATTR_TIMEOUT => 30
        ];
        
        $this->connect();
    }
    
    public static function getInstance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        
        return self::$instance;
    }
    
    private function connect(): void
    {
        $dsn = "mysql:host={$this->host};dbname={$this->database};charset={$this->charset}";
        
        try {
            $this->connection = new PDO($dsn, $this->username, $this->password, $this->options);
            
            // Set timezone
            $this->connection->exec("SET time_zone = '+00:00'");
            
        } catch (PDOException $e) {
            $this->handleConnectionError($e);
        }
    }
    
    public function getConnection(): PDO
    {
        // Check if connection is still alive
        if (!$this->isConnected()) {
            $this->connect();
        }
        
        return $this->connection;
    }
    
    private function isConnected(): bool
    {
        try {
            $this->connection->query('SELECT 1');
            return true;
        } catch (PDOException $e) {
            return false;
        }
    }
    
    public function beginTransaction(): bool
    {
        return $this->connection->beginTransaction();
    }
    
    public function commit(): bool
    {
        return $this->connection->commit();
    }
    
    public function rollback(): bool
    {
        return $this->connection->rollback();
    }
    
    public function inTransaction(): bool
    {
        return $this->connection->inTransaction();
    }
    
    public function lastInsertId(): string
    {
        return $this->connection->lastInsertId();
    }
    
    public function prepare(string $statement, array $options = []): \PDOStatement
    {
        return $this->connection->prepare($statement, $options);
    }
    
    public function query(string $statement): \PDOStatement
    {
        return $this->connection->query($statement);
    }
    
    public function exec(string $statement): int
    {
        return $this->connection->exec($statement);
    }
    
    public function quote(string $string, int $type = PDO::PARAM_STR): string
    {
        return $this->connection->quote($string, $type);
    }
    
    private function handleConnectionError(PDOException $e): void
    {
        $errorMessage = "Database connection failed: " . $e->getMessage();
        
        // Log error
        error_log($errorMessage);
        
        // In production, show generic error
        if (Config::isProduction()) {
            throw new \Exception('Database connection failed. Please try again later.');
        }
        
        // In development, show detailed error
        throw new \Exception($errorMessage);
    }
    
    public function testConnection(): bool
    {
        try {
            $stmt = $this->connection->query('SELECT 1 as test');
            $result = $stmt->fetch();
            return $result['test'] === 1;
        } catch (PDOException $e) {
            return false;
        }
    }
    
    public function getServerInfo(): array
    {
        return [
            'version' => $this->connection->getAttribute(PDO::ATTR_SERVER_VERSION),
            'driver' => $this->connection->getAttribute(PDO::ATTR_DRIVER_NAME),
            'client_version' => $this->connection->getAttribute(PDO::ATTR_CLIENT_VERSION),
            'connection_status' => $this->connection->getAttribute(PDO::ATTR_CONNECTION_STATUS),
            'server_info' => $this->connection->getAttribute(PDO::ATTR_SERVER_INFO)
        ];
    }
    
    public function getTables(): array
    {
        $stmt = $this->connection->query("SHOW TABLES");
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
    
    public function tableExists(string $tableName): bool
    {
        $stmt = $this->connection->prepare("SHOW TABLES LIKE ?");
        $stmt->execute([$tableName]);
        return $stmt->rowCount() > 0;
    }
    
    public function getTableColumns(string $tableName): array
    {
        $stmt = $this->connection->prepare("DESCRIBE {$tableName}");
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    // Prevent cloning
    private function __clone() {}
    
    // Prevent unserialization
    public function __wakeup()
    {
        throw new \Exception("Cannot unserialize singleton");
    }
}