<?php

namespace SmartForm\Config;

class Config
{
    private static $config = null;
    
    public static function load(): void
    {
        if (self::$config !== null) {
            return;
        }
        
        // Load environment variables
        self::loadEnvironment();
        
        // Load configuration arrays
        self::$config = [
            'app' => require_once __DIR__ . '/../../config/app.php',
            'database' => require_once __DIR__ . '/../../config/database.php',
            'mail' => require_once __DIR__ . '/../../config/mail.php',
            'services' => require_once __DIR__ . '/../../config/services.php'
        ];
    }
    
    public static function get(string $key, $default = null)
    {
        if (self::$config === null) {
            self::load();
        }
        
        $keys = explode('.', $key);
        $value = self::$config;
        
        foreach ($keys as $k) {
            if (!isset($value[$k])) {
                return $default;
            }
            $value = $value[$k];
        }
        
        return $value;
    }
    
    public static function set(string $key, $value): void
    {
        if (self::$config === null) {
            self::load();
        }
        
        $keys = explode('.', $key);
        $config = &self::$config;
        
        foreach ($keys as $k) {
            if (!isset($config[$k])) {
                $config[$k] = [];
            }
            $config = &$config[$k];
        }
        
        $config = $value;
    }
    
    private static function loadEnvironment(): void
    {
        $envFile = __DIR__ . '/../../.env';
        
        if (!file_exists($envFile)) {
            throw new \Exception('.env file not found');
        }
        
        $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        
        foreach ($lines as $line) {
            if (strpos(trim($line), '#') === 0) {
                continue; // Skip comments
            }
            
            list($name, $value) = explode('=', $line, 2);
            $name = trim($name);
            $value = trim($value);
            
            // Remove quotes if present
            if (preg_match('/^"(.*)"$/', $value, $matches)) {
                $value = $matches[1];
            } elseif (preg_match("/^'(.*)'$/", $value, $matches)) {
                $value = $matches[1];
            }
            
            $_ENV[$name] = $value;
            putenv("$name=$value");
        }
    }
    
    public static function env(string $key, $default = null)
    {
        $value = $_ENV[$key] ?? getenv($key);
        
        if ($value === false) {
            return $default;
        }
        
        // Convert string booleans
        if (strtolower($value) === 'true') {
            return true;
        }
        
        if (strtolower($value) === 'false') {
            return false;
        }
        
        // Convert string null
        if (strtolower($value) === 'null') {
            return null;
        }
        
        return $value;
    }
    
    public static function isProduction(): bool
    {
        return self::env('APP_ENV', 'production') === 'production';
    }
    
    public static function isDebug(): bool
    {
        return self::env('APP_DEBUG', false) === true;
    }
    
    public static function getAppUrl(): string
    {
        return rtrim(self::env('APP_URL', 'http://localhost'), '/');
    }
    
    public static function getAppKey(): string
    {
        $key = self::env('APP_KEY');
        
        if (empty($key)) {
            throw new \Exception('APP_KEY not set in environment');
        }
        
        return $key;
    }
}

// Auto-load configuration
Config::load();